using System;
using  System.ComponentModel;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using ArrayList = System.Collections.ArrayList;
using Cursor    = System.Windows.Forms.Cursor;
using DynaPDF;

namespace PDFControl
{
   public class ErrorArgs : System.EventArgs
   {
      public ErrorArgs(System.Collections.ArrayList e)
      {
         Errors = e;
      }
      public System.Collections.ArrayList Errors;
   }

   public class NewPageArgs : System.EventArgs
   {
      public NewPageArgs(int Page)
      {
         PageNum = Page;
      }
      public int PageNum;
   }

   public class UpdateScreenArgs : System.EventArgs
   {
      public UpdateScreenArgs(IntPtr DC_, TIntRect Area_)
      {
         Area = Area_;
         DC   = DC_;
      }
      public TIntRect Area;
      public IntPtr   DC;
   }

   public partial class PDFCanvas : UserControl
   {
      public delegate void ErrorHandler(object sender, ErrorArgs e);
      public delegate void NewPageHandler(object sender, NewPageArgs e);
      public delegate void OnAfterUpdateScreenHandler(object sender, UpdateScreenArgs e);

      /*
         The error callback function works differently with the page cache. It does not return error messages since all error
         messages are stored in the error log. The error callback function is called when it is safe to access the error log
         and if new messages are available.

         The parameter ErrCode is set to the page number that produced the error or to -1 if the error occurred during loading
         the top level objects.

         The rendering thread (if running) waits until the error callback function returns. So, it is not required to synchronize
         anything since no competing threads are running.
      */
      private int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         int i, count;
         count = m_PDF.GetErrLogMessageCount();
         if (m_Errors.Count + count > m_MaxErrCount)
            count = m_MaxErrCount - m_Errors.Count;
         if (count < 1) return 0;
         TPDFError msg = new TPDFError();
         try
         {
            for (i = 0; i < count; i++)
            {
               if (m_PDF.GetErrLogMessage(i, ref msg))
               {
                  // ErrCode is set to the page number if the errors occurred during rendering a page!
                  if (ErrCode > 0)
                     m_Errors.Add(String.Format("Page {0}: {1}, ObjNum: {2}, Offset: {3}", ErrCode, msg.Message, msg.ObjNum, msg.Offset));
                  else
                     m_Errors.Add(String.Format("{0}, ObjNum: {1}, Offset: {2}", msg.Message, msg.ObjNum, msg.Offset));
               }
            }
            // Clear the error log so that we don't receive the same messages again
            m_PDF.ClearErrorLog();
            if (Error != null)
               BeginInvoke(Error, this, new ErrorArgs(m_Errors));
         }catch
         {
            return -1;
         }
         return 0;
      }

      /*
         This callback is excuted from the rendering thread whenever the screen was updated. The rendering
         thread waits until the function returns.

         However, note that the rendering thread calls this function and not the main thread! Be careful
         what you do here since UI components are not necessarily thread-safe.

         If you want to draw additional controls on the canvas then draw them in the OnAfterUpdateScreen event.
         The provided dc is the one of the PDFCanvas. This is a private device context. Changes made on this dc
         are not restored among paint events! So, make sure that you restore important changes, e.g. the coordinate
         system or clipping region if necessary.
       */

      [method: System.Diagnostics.CodeAnalysis.SuppressMessage("Reliability", "IDE1005:Delegate invocation can be simplified", Scope = "module")]
      private int OnUpdateWindow(IntPtr Data, ref TIntRect Area)
      {
         if (OnAfterUpdateScreen != null)
            OnAfterUpdateScreen(this, new UpdateScreenArgs(this.GetDC(), Area));
         return 0;
      }

      protected override CreateParams CreateParams
      {
         get
         {
            CreateParams cp = base.CreateParams;
            // We need a private dc!
            cp.Style      |= (0x00100000 | 0x00200000 | 0x00010000); // WS_HSCROLL | WS_VSCROLL | WS_TABSTOP
            cp.ClassStyle &= ~(0x0040 | 0x0080);                     // CS_CLASSDC, CS_PARENTDC
            cp.ClassStyle |= 0x0020;                                 // CS_OWNDC
            return cp;
         }
      }

      public PDFCanvas()
      {
         m_AutoScroll      = true;
         m_ColorManagement = true;
         m_MaxErrCount     = 100;
         m_ScrollWindow    = true;
         InitializeComponent();
         this.SetStyle(ControlStyles.UserPaint | ControlStyles.AllPaintingInWmPaint | ControlStyles.ResizeRedraw | ControlStyles.Opaque, true);
         this.UpdateStyles();

         m_DC         = GetDC(this.Handle);
         m_PageCount  = 0;
         m_Resolution = GetDeviceCaps(m_DC, LOGPIXELSX);
         InitScrollBar(true, 0, 0, 0);
         InitScrollBar(false, 0, 0, 0);
         if (!DesignMode)
         {
            m_PDF    = new CPDF();
            m_Errors = new System.Collections.ArrayList(m_MaxErrCount);
            m_PDF.SetErrorMode((TErrMode)m_PDF.GetErrorMode() | TErrMode.emUseErrLog);
            m_PDF.SetOnErrorProc(IntPtr.Zero, new TErrorProc(PDFError));
            // Min version 4.0.18.45
            if (m_PDF.GetDynaPDFVersionInt() < 40180045)
               throw new Exception(String.Format("Wrong dynapdf.dll version!\nFound {0}\nRequire 3.0.9.xxx", m_PDF.GetDynaPDFVersion()));

            m_Cache = CPDF.rasCreatePageCache(m_PDF.GetPDFInstance(), TPDFPixFormat.pxfBGRA, 4, 8, (uint)((BackColor.R | (BackColor.G << 8) | (BackColor.B << 16))));
            // Very important: We must store the pointer of the callback function in a member variable, otherwise the
            // garbage collector deletes the function very soon since it means it is no longer required. Very intelligent!
            m_OnPaintCallback = new TOnUpdateWindow(OnUpdateWindow);
            CPDF.rasSetOnPaintCallback(m_Cache, IntPtr.Zero, m_OnPaintCallback);

            CPDF.rasInitialize(m_Cache, TPDFThreadPriority.ttpLowest);
            CPDF.rasResize(m_Cache, ClientSize.Width, ClientSize.Height);
            System.Reflection.Assembly a = System.Reflection.Assembly.GetExecutingAssembly();
            m_HandClosed = new Cursor(a.GetManifestResourceStream("DynaPDF.Resources.hand_closed.cur"));
            m_HandNormal = new Cursor(a.GetManifestResourceStream("DynaPDF.Resources.hand_normal.cur"));
         }
         if (m_Resolution < 72) m_Resolution = 72;
      }

      protected override void Dispose(bool disposing)
      {
         if (disposing && (components != null))
         {
            components.Dispose();
         }
         base.Dispose(disposing);
         if (!IntPtr.Zero.Equals(m_Cache)) CPDF.rasDeletePageCache(ref m_Cache);
         m_PDF = null;
      }

      // -----------------------------------------------------------------------------------------------------------------------------------------------------------

      public void AddError(string ErrMessage)
      {
         m_Errors.Add(ErrMessage);
         if (Error != null) BeginInvoke(Error, this, new ErrorArgs(m_Errors));
      }

      public override bool AutoScroll
      {
         get
         {
            return m_AutoScroll;
         }
         set
         {
            m_AutoScroll = value;
            ShowScrollBar(Handle, SB_BOTH, Convert.ToInt32(m_AutoScroll));
         }
      }

      public override System.Drawing.Color BackColor
      {
         get
         {
            return base.BackColor;
         }
         set
         {
            base.BackColor = value;
            if (!IntPtr.Zero.Equals(m_Cache)) CPDF.rasChangeBackColor(m_Cache, (uint)((BackColor.R | (BackColor.G << 8) | (BackColor.B << 16))));
         }
      }

      public void CloseFile()
      {
         CPDF.rasCloseFile(m_Cache);
         m_Errors.Clear();
         Cursor      = Cursors.Arrow;
         m_FirstPage = 0;
         m_PageCount = 0;
         DisableScrollBars();
         Invalidate();
      }

      [Browsable(true)]
      public bool ColorManagement
      {
         get
         {
            return m_ColorManagement;
         }
         set
         {
            m_ColorManagement = value;
         }
      }

      protected void DisableScrollBars()
      {
         TSCROLLINFO si = new TSCROLLINFO();
         si.cbSize = System.Runtime.InteropServices.Marshal.SizeOf(si);
         si.fMask  = SIF_ALL;
         SetScrollPos(true, 0, FALSE);
         SetScrollPos(false, 0, FALSE);
         SetScrollInfo(Handle, SB_BOTH, si, TRUE);
         SetScrollInfo(Handle, SB_VERT, si, TRUE);
         ShowScrollBar(Handle, SB_VERT, TRUE);
         ShowScrollBar(Handle, SB_HORZ, TRUE);
         EnableScrollBar(Handle, SB_VERT, ESB_DISABLE_BOTH);
         EnableScrollBar(Handle, SB_HORZ, ESB_DISABLE_BOTH);
      }

      public void DisplayFirstPage()
      {
         if (!IntPtr.Zero.Equals(m_Cache))
         {
            int pos = CPDF.rasGetScrollPos(m_Cache, TRUE, m_FirstPage);
            CPDF.rasResetMousePos(m_Cache);
            SetScrollPos(true, pos, TRUE);
            Invalidate();
         }
      }

      [Browsable(true)]
      public event ErrorHandler Error;

      public TUpdBmkAction ExecBookmark(int Index)
      {
         Single z = 0.0f;
         int x = 0, y = 0;
         m_OldPage = 0;
         TPDFPageScale ps = TPDFPageScale.psFitWidth;
         TUpdBmkAction retval = CPDF.rasExecBookmark(m_Cache, Index, ref x, ref y, ref z, ref ps, null);
         if ((retval & TUpdBmkAction.ubaOpenPage) == TUpdBmkAction.ubaOpenPage)
         {
            if ((retval & TUpdBmkAction.ubaZoom) == TUpdBmkAction.ubaZoom)
            {
               int max = 0, small = 0, large = 0;
               CPDF.rasZoom(m_Cache, z * 72.0f / m_Resolution, ref x, ref y);
               m_ZoomMode = true;
               CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
               SetScrollPos(true, y, TRUE);
               if (CPDF.rasGetScrollRange(m_Cache, FALSE, ref max, ref small, ref large) != 0)
               {
                  InitScrollBar(false, max, small, large);
                  SetScrollPos(false, x, TRUE);
               }else
               {
                  SetScrollPos(false, 0, TRUE);
                  EnableScrollBar(Handle, SB_HORZ, ESB_DISABLE_BOTH);
               }
            }else
            {
               if ((retval & TUpdBmkAction.ubaPageScale) == TUpdBmkAction.ubaPageScale)
               {
                  if (m_ZoomMode || ps != CPDF.rasGetPageScale(m_Cache))
                  {
                     m_ZoomMode = false;
                     CPDF.rasSetPageScale(m_Cache, ps);
                     UpdateScrollBarsEx();
                  }
               }
               SetScrollPos(true, y, TRUE);
               SetScrollPos(false, x, TRUE);
            }
         }
         Invalidate();
         return retval;
      }

      public int FirstPage
      {
         get{return m_FirstPage;}
      }

      public IntPtr GetDC()
      {
         return m_DC;
      }

      public IntPtr GetCacheInstance()
      {
         return m_Cache;
      }

       // The function returns the page number at the curser position on success, as well the exact position and size of the rendered page.
       // You must check whether the return value is greater 0. Otherwise there is no page at the cursor coordinates or the page was not loaded yet.
       // The x-coordinate of the cursor is currently not taken into account because only one page can occur in horizontal direction at this time.
       // The matrix transforms PDF space to device space. DestX/Y must be added to get final device coordinates.
      public int GetPageMatrix(int CursorX, int CursorY, ref int DestX, ref int DestY, ref int Width, ref int Height, ref TCTM Matrix)
      {
         return CPDF.rasGetPageMatrix(m_Cache, CursorX, CursorY, ref DestX, ref DestY, ref Width, ref Height, ref Matrix);
      }

      public CPDF GetPDFInstance()
      {
         return m_PDF;
      }

      public int GetScrollLineDelta(bool Vertical)
      {
         return CPDF.rasGetScrollLineDelta(m_Cache, Convert.ToInt32(Vertical));
      }

      protected int GetScrollPos(bool Vertical)
      {
         TSCROLLINFO si = new TSCROLLINFO();
         si.cbSize = System.Runtime.InteropServices.Marshal.SizeOf(si);
         si.fMask  = SIF_POS;
         si.nPos   = 0;
         if (Vertical)
            GetScrollInfo(Handle, SB_VERT, si);
         else
            GetScrollInfo(Handle, SB_HORZ, si);
         return si.nPos;
      }

      protected int GetThumbPos(bool Vertical)
      {
         TSCROLLINFO si = new TSCROLLINFO();
         si.cbSize = System.Runtime.InteropServices.Marshal.SizeOf(si);
         si.fMask  = SIF_TRACKPOS;
         si.nPos   = 0;
         if (Vertical)
            GetScrollInfo(Handle, SB_VERT, si);
         else
            GetScrollInfo(Handle, SB_HORZ, si);
         return si.nTrackPos;
      }

      public bool InitBaseObjects(TInitCacheFlags Flags)
      {
         if (IntPtr.Zero.Equals(m_Cache)) return false;
         m_Errors.Clear();
         m_HavePos   = false;
         m_OldPage   = 0;
         m_ZoomMode  = false;
         m_PageCount = m_PDF.GetInPageCount();
         // InitBaseObjects() opens the PDF file again and hence produces the same error messages for the top level objects.
         // To avoid double error messages we clear the error log of the base instance.
         m_PDF.ClearErrorLog();
         m_FirstPage = CPDF.rasInitBaseObjects(m_Cache, ClientSize.Width, ClientSize.Height, Flags);
         if (m_FirstPage < 1 || m_PageCount < 1) return false;
         if (m_AutoScroll)
         {
            int max = 0, small = 0, large = 0;
            CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
            InitScrollBar(true, max, small, large);
            InitScrollBar(false, 0, 0, 0);
            small = CPDF.rasGetScrollPos(m_Cache, TRUE, m_FirstPage);
            SetScrollPos(true, small, TRUE);
         }
         Cursor = m_HandNormal;
         return true;
      }

      protected void InitScrollBar(bool Vertical, int Max, int SmallChange, int LargeChange)
      {
         TSCROLLINFO si = new TSCROLLINFO();
         si.cbSize = System.Runtime.InteropServices.Marshal.SizeOf(si);
         si.fMask  = SIF_POS | SIF_RANGE | SIF_PAGE | SIF_DISABLENOSCROLL;
         si.nMin   = 0;
         si.nMax   = Max;
         si.nPage  = LargeChange;
         si.nPos   = 0;
         if (Vertical)
         {
            m_ScrollVMax = Max - LargeChange + 1;
            SetScrollInfo(Handle, SB_VERT, si, FALSE);
         }else
         {
            m_ScrollHMax = Max - LargeChange + 1;
            SetScrollInfo(Handle, SB_HORZ, si, FALSE);
         }
      }

      [Browsable(true)]
      public int MaxErrCount
      {
         get
         {
            return m_MaxErrCount;
         }
         set
         {
            m_MaxErrCount = value;
         }
      }

      [Browsable(true)]
      public event NewPageHandler NewPage;

      [Browsable(true)]
      public event OnAfterUpdateScreenHandler OnAfterUpdateScreen;

      protected override void OnMouseDown(MouseEventArgs e)
      {
         if (m_PageCount == 0)
            Cursor = Cursors.Arrow;
         else
         {
            m_HavePos = true;
            switch(CPDF.rasMouseDown(m_Cache, e.X, e.Y))
            {
               case TPDFCursor.pcrHandClosed: Cursor = m_HandClosed; break;
               case TPDFCursor.pcrHandPoint:  Cursor = Cursors.Hand; break;
               default:                       break;
            }
         }
         base.OnMouseDown(e);
      }

      protected override void OnMouseMove(MouseEventArgs e)
      {
         if (m_PageCount > 0 && m_HavePos)
         {
            int sx, sy;
            TUpdScrollbar retval;
            if (e.Button == MouseButtons.None)
            {
               sx = GetScrollPos(false);
               sy = GetScrollPos(true);
               retval = CPDF.rasMouseMove(m_Cache, IntPtr.Zero, FALSE, ref sx, ref sy, e.X, e.Y);
               switch(retval & TUpdScrollbar.usbCursorMask)
               {
                  case TUpdScrollbar.usbCursorHandNormal: Cursor = m_HandNormal;  break;
                  case TUpdScrollbar.usbCursorHandClosed: Cursor = m_HandClosed;  break;
                  case TUpdScrollbar.usbCursorHandPoint:  Cursor = Cursors.Hand;  break;
                  case TUpdScrollbar.usbCursorIBeam:      Cursor = Cursors.IBeam; break;
                  default:                                break;
               }
            }else if (e.Button == MouseButtons.Left)
            {
               sx = GetScrollPos(false);
               sy = GetScrollPos(true);
               if (m_ScrollWindow)
                  retval = CPDF.rasMouseMove(m_Cache, Handle, TRUE, ref sx, ref sy, e.X, e.Y);
               else
                  retval = CPDF.rasMouseMove(m_Cache, IntPtr.Zero, FALSE, ref sx, ref sy, e.X, e.Y);
               if ((retval & TUpdScrollbar.usbVertRange) != 0)
               {
                  int max = 0, small = 0, large = 0;
                  CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
                  InitScrollBar(true, max, small, large);
               }
               if ((retval & TUpdScrollbar.usbHorzRange) != 0)
               {
                  int max = 0, small = 0, large = 0;
                  CPDF.rasGetScrollRange(m_Cache, FALSE, ref max, ref small, ref large);
                  InitScrollBar(false, max, small, large);
               }
               switch(retval & TUpdScrollbar.usbCursorMask)
               {
                  case TUpdScrollbar.usbCursorHandNormal: Cursor = m_HandNormal;  break;
                  case TUpdScrollbar.usbCursorHandClosed: Cursor = m_HandClosed;  break;
                  case TUpdScrollbar.usbCursorHandPoint:  Cursor = Cursors.Hand;  break;
                  case TUpdScrollbar.usbCursorIBeam:      Cursor = Cursors.IBeam; break;
                  default:                                                        break;
               }
               if (!m_ScrollWindow) Invalidate();
               SetScrollPos(false, sx, TRUE);
               SetScrollPos(true, sy, TRUE);
            }
         }
         base.OnMouseMove(e);
      }

      protected override void OnMouseUp(MouseEventArgs e)
      {
         m_HavePos = false;
         if (m_PageCount > 0)
            Cursor = m_HandNormal;
         base.OnMouseUp(e);
      }

      protected override void OnMouseWheel(MouseEventArgs e)
      {
         int x, y, gap;
         TUpdScrollbar update;
         if (!m_AutoScroll || IntPtr.Zero.Equals(m_Cache)) return;

         gap = e.Delta * CPDF.rasGetScrollLineDelta(m_Cache, TRUE) / 120;

         x = GetScrollPos(false);
         y = GetScrollPos(true) - gap;
         if (y < 0) y = 0;

         update = CPDF.rasScroll(m_Cache, TRUE, SB_THUMBTRACK, ref x, ref y);
         
         if ((update & TUpdScrollbar.usbVertRange) != 0)
         {
            int max = 0, smal = 0, large = 0;
            CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref smal, ref large);
            InitScrollBar(true, max, smal, large);
         }
         if ((update & TUpdScrollbar.usbHorzRange) != 0)
         {
            int max = 0, smal = 0, large = 0;
            CPDF.rasGetScrollRange(m_Cache, FALSE, ref max, ref smal, ref large);
            InitScrollBar(false, max, smal, large);
         }
         SetScrollPos(true, y, Convert.ToInt32(update));

         Invalidate();

         base.OnMouseWheel(e);
      }

      [method: System.Diagnostics.CodeAnalysis.SuppressMessage("Reliability", "IDE0017:Initialization can be simplified", Scope = "module")]
      [method: System.Diagnostics.CodeAnalysis.SuppressMessage("Reliability", "IDE1005:Delegate invocation can be simplified", Scope = "module")]
      protected override void OnPaint(PaintEventArgs e)
      {
         if (IntPtr.Zero.Equals(m_Cache))
         {
            TRect r = new TRect();
            r.Left   = ClientRectangle.Left;
            r.Right  = ClientRectangle.Right;
            r.Bottom = ClientRectangle.Bottom;
            r.Top    = ClientRectangle.Top;
            SetBkColor(m_DC, (uint)((BackColor.R | (BackColor.G << 8) | (BackColor.B << 16))));
            ExtTextOut(m_DC, 0, 0, ETO_OPAQUE, r, null, 0, IntPtr.Zero);
         }else
         {
            int x = GetScrollPos(false);
            int y = GetScrollPos(true);
            TUpdScrollbar retval  = CPDF.rasPaint(m_Cache, m_DC, ref x, ref y);
            int pageNum = CPDF.rasGetCurrPage(m_Cache);
            if (retval != TUpdScrollbar.usbNoUpdate)
            {
               if ((retval & TUpdScrollbar.usbVertRange) != 0)
               {
                  int max = 0, small = 0, large = 0;
                  CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
                  InitScrollBar(true, max, small, large);
               }
               if ((retval & TUpdScrollbar.usbHorzRange) != 0)
               {
                  int max = 0, small = 0, large = 0;
                  CPDF.rasGetScrollRange(m_Cache, FALSE, ref max, ref small, ref large);
                  InitScrollBar(false, max, small, large);
                  SetScrollPos(false, x, TRUE);
               }else
               {
                  SetScrollPos(false, 0, FALSE);
                  EnableScrollBar(Handle, SB_HORZ, ESB_DISABLE_BOTH);
               }
               SetScrollPos(true, y, TRUE);
            }
            if (m_OldPage != pageNum)
            {
               m_OldPage = pageNum;
               if (NewPage != null) NewPage(this, new NewPageArgs(pageNum));
            }
         }
         base.OnPaint(e);
      }

      protected override void OnResize(EventArgs e)
      {
         if (!m_Initialized)
         {
            m_Initialized = true;
            if (m_AutoScroll)
            {
               ShowScrollBar(Handle, SB_BOTH, TRUE);
               EnableScrollBar(Handle, SB_BOTH, ESB_DISABLE_BOTH);
            }else
            {
               IntPtr saved;
               // Hidding the scrollbars causes two new resize events.
               saved   = m_Cache;
               m_Cache = IntPtr.Zero;
               ShowScrollBar(Handle, SB_BOTH, FALSE);
               m_Cache = saved;
            }
         }
         if (!IntPtr.Zero.Equals(m_Cache))
         {
            m_OldPage = 0;
            int page = CPDF.rasGetCurrPage(m_Cache);
            CPDF.rasResize(m_Cache, ClientSize.Width, ClientSize.Height);
            // If AutoScroll is disabled then you must initialize your scroll bars in the very same way in
            // the OnResize event.
            if (m_AutoScroll)
            {
               int pos, max = 0, small = 0, large = 0;
               pos = CPDF.rasGetScrollPos(m_Cache, TRUE, page);
               CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
               InitScrollBar(true, max, small, large);
               SetScrollPos(true, pos, TRUE);

               CPDF.rasGetScrollRange(m_Cache, FALSE, ref max, ref small, ref large);
               InitScrollBar(false, max, small, large);
               Invalidate();
            }
         }
      }

      protected override void OnScroll(ScrollEventArgs e)
      {
         if (m_AutoScroll && !IntPtr.Zero.Equals(m_Cache))
         {
            if (e.ScrollOrientation == ScrollOrientation.VerticalScroll)
            {
               int x = 0, y = GetScrollPos(true);
               switch (e.Type)
               {
                  case ScrollEventType.First:      y = 0;                  break;
                  case ScrollEventType.Last:       y = m_ScrollVMax;       break;
                  case ScrollEventType.ThumbTrack: y = GetThumbPos(true);  break;
                  default:                         y = GetScrollPos(true); break;
                  case ScrollEventType.EndScroll:  return;
               }
               CPDF.rasScroll(m_Cache, TRUE, (int)e.Type, ref x, ref y);
               SetScrollPos(true, y, TRUE);
            }else
            {
               int x = GetScrollPos(false), y = 0;
               switch (e.Type)
               {
                  case ScrollEventType.First:      x = 0;                   break;
                  case ScrollEventType.Last:       x = m_ScrollHMax;        break;
                  case ScrollEventType.ThumbTrack: x = GetThumbPos(false);  break;
                  default:                         x = GetScrollPos(false); break;
                  case ScrollEventType.EndScroll:  return;
               }
               CPDF.rasScroll(m_Cache, FALSE, (int)e.Type, ref x, ref y);
               SetScrollPos(false, x, TRUE);
            }
            Invalidate();
         }
         base.OnScroll(e);
      }

      public int PageCount
      {
         get{return m_PageCount;}
      }

      public TPageLayout PageLayout
      {
         get
         {
            return CPDF.rasGetPageLayout(m_Cache);
         }
         set
         {
            if (value != CPDF.rasGetPageLayout(m_Cache))
            {
               CPDF.rasSetPageLayout(m_Cache, value);
               if (value == TPageLayout.plSinglePage)
               {
                  CPDF.rasResetMousePos(m_Cache);
                  int pos = CPDF.rasGetScrollPos(m_Cache, TRUE, CPDF.rasGetCurrPage(m_Cache));
                  SetScrollPos(true, pos, TRUE);
               }
               Invalidate();
            }
         }
      }

      public TPDFPageScale PageScale
      {
         get
         {
            return CPDF.rasGetPageScale(m_Cache);
         }
         set
         {
            if (m_ZoomMode || value != CPDF.rasGetPageScale(m_Cache))
            {
               m_OldPage  = 0;
               m_ZoomMode = false;
               CPDF.rasSetPageScale(m_Cache, value);
               CPDF.rasResetMousePos(m_Cache);
               UpdateScrollBars();
            }
         }
      }

      public void ProcessErrors(bool UpdateWindow)
      {
         // If we have errors in the pipe then ProcessErrors() stops the rendering thread and calls then
         // the error callback function so that we can safely access the error log. If the return value
         // is true then the error callback function was called and the rendering thread was stopped.
         if (CPDF.rasProcessErrors(m_Cache) != 0 && UpdateWindow)
            Invalidate();
      }

      protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
      {
         int pos;
         switch(msg.WParam.ToInt32())
         {
            case VK_DOWN:  SendMessage(msg.HWnd, WM_VSCROLL, SB_LINEDOWN,  0); break;
            case VK_UP:    SendMessage(msg.HWnd, WM_VSCROLL, SB_LINEUP,    0); break;
            case VK_RIGHT: SendMessage(msg.HWnd, WM_HSCROLL, SB_LINELEFT,  0); break;
            case VK_LEFT:  SendMessage(msg.HWnd, WM_HSCROLL, SB_LINERIGHT, 0); break;
            // Note that the scroll messages SB_PAGEUP or SB_PAGEDOWN scroll maybe more than one page up or down. So, we use
            // the scroll message only if the file contains one page. Otherwise we calculate the scroll position and scroll
            // one page forward or backward as needed.
            case VK_PRIOR:
            {
               if (CPDF.rasGetCurrPage(m_Cache) > 1)
               {
                  pos = CPDF.rasGetScrollPos(m_Cache, TRUE, CPDF.rasGetCurrPage(m_Cache) - 1);
                  CPDF.rasResetMousePos(m_Cache);
                  SetScrollPos(true, pos, TRUE);
                  Invalidate();
               }else
                  SendMessage(Handle, WM_VSCROLL, SB_PAGEUP, 0);
               break;
            }
            case VK_NEXT:
            {
               CPDF.rasResetMousePos(m_Cache);
               if (CPDF.rasGetCurrPage(m_Cache) < m_PageCount)
               {
                  pos = CPDF.rasGetScrollPos(m_Cache, TRUE, CPDF.rasGetCurrPage(m_Cache) + 1);
                  SetScrollPos(true, pos, TRUE);
                  Invalidate();
               }else
                  SendMessage(Handle, WM_VSCROLL, SB_PAGEDOWN, 0);
               break;
            }
            case VK_HOME:
            {
               CPDF.rasResetMousePos(m_Cache);
               pos = CPDF.rasGetScrollPos(m_Cache, TRUE, 1);
               SetScrollPos(true, pos, TRUE);
               Invalidate();
               break;
            }
            case VK_END:
            {
               CPDF.rasResetMousePos(m_Cache);
               if (m_PageCount > 1)
                  pos = CPDF.rasGetScrollPos(m_Cache, TRUE, m_PageCount);
               else
                  pos = m_ScrollVMax;
               SetScrollPos(true, pos, TRUE);
               Invalidate();
               break;
            }
            default: break;
         }
         return base.ProcessCmdKey(ref msg, keyData);
      }

      public int Resolution
      {
         get{return m_Resolution;}
      }

      public int Rotate
      {
         get
         {
            return CPDF.rasGetRotate(m_Cache);
         }
         set
         {
            int pos, max = 0, small = 0, large = 0;
            m_OldPage = 0; // Make sure that a NewPage event will be raised
            // We must update the scroll bars when we rotate the pages
            pos = CPDF.rasGetScrollPos(m_Cache, TRUE, CPDF.rasGetCurrPage(m_Cache));
            CPDF.rasSetRotate(m_Cache, value);

            CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
            CPDF.rasResetMousePos(m_Cache);
            InitScrollBar(true, max, small, large);

            CPDF.rasGetScrollRange(m_Cache, FALSE, ref max, ref small, ref large);
            InitScrollBar(false, max, small, large);
            SetScrollPos(true, pos, TRUE);
            Invalidate();
         }
      }

      public void SetDefPageLayout(TPageLayout Value)
      {
         CPDF.rasSetDefPageLayout(m_Cache, Value);
      }

      // The function changes the state in the main and render PDF instance.
      public bool SetOCGState(int Handle, bool On, bool SaveState)
      {
         if (CPDF.rasSetOCGState(m_Cache, Handle, Convert.ToInt32(On), Convert.ToInt32(SaveState)) != 0)
         {
            Invalidate();
            return true;
         }else
            return false;
      }

      public bool SetScrollLineDelta(bool Vertical, int Value)
      {
         int y = GetScrollPos(true);
         if (CPDF.rasSetScrollLineDelta(m_Cache, Convert.ToInt32(Vertical), Value) != 0)
         {
            int max = 0, small = 0, large = 0;
            CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
            InitScrollBar(true, max, small, large);
            SetScrollPos(true, y, TRUE);
            return true;
         }
         return false;
      }

      protected void SetScrollPos(bool Vertical, int Value, int Redraw)
      {
         TSCROLLINFO si = new TSCROLLINFO();
         si.cbSize = System.Runtime.InteropServices.Marshal.SizeOf(si);
         si.fMask  = SIF_POS;
         si.nPos   = Value;
         if (Vertical)
            SetScrollInfo(Handle, SB_VERT, si, Redraw);
         else
            SetScrollInfo(Handle, SB_HORZ, si, Redraw);
      }

      public void ScrollTo(int PageNum)
      {
         if (!IntPtr.Zero.Equals(m_Cache))
         {
            int y = CPDF.rasGetScrollPos(m_Cache, TRUE, PageNum);
            CPDF.rasResetMousePos(m_Cache);
            SetScrollPos(true, y, TRUE);
            Invalidate();
         }
      }

      protected void UpdateScrollBars()
      {
         int pageNum = CPDF.rasGetCurrPage(m_Cache);
         UpdateScrollBarsEx();
         SetScrollPos(false, 0, TRUE);
         ScrollTo(pageNum);
      }

      protected void UpdateScrollBarsEx()
      {
         int max = 0, small = 0, large = 0;
         CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
         InitScrollBar(true, max, small, large);

         CPDF.rasGetScrollRange(m_Cache, FALSE, ref max, ref small, ref large);
         InitScrollBar(false, max, small, large);
      }

      public bool UseScrollWindow
      {
         get
         {
            return m_ScrollWindow;
         }
         set
         {
            m_ScrollWindow = value;
         }
      }

      public void Zoom(Single Value)
      {
         int x, y, max = 0, small = 0, large = 0;
         x = GetScrollPos(false);
         y = GetScrollPos(true);
         CPDF.rasZoom(m_Cache, Value, ref x, ref y);

         CPDF.rasGetScrollRange(m_Cache, TRUE, ref max, ref small, ref large);
         InitScrollBar(true, max, small, large);
         SetScrollPos(true, y, TRUE);
         if (CPDF.rasGetScrollRange(m_Cache, FALSE, ref max, ref small, ref large) != 0)
         {
            InitScrollBar(false, max, small, large);
            SetScrollPos(false, x, TRUE);
         }else
         {
            SetScrollPos(false, 0, TRUE);
            EnableScrollBar(Handle, SB_HORZ, ESB_DISABLE_BOTH);
         }
         m_OldPage  = 0;
         m_ZoomMode = true;
         Invalidate();
      }

      private bool      m_AutoScroll;
      private IntPtr    m_Cache;
      private bool      m_ColorManagement;
      private IntPtr    m_DC;
      private ArrayList m_Errors;
      private int       m_FirstPage;
      private Cursor    m_HandClosed;
      private Cursor    m_HandNormal;
      private bool      m_HavePos;
      private bool      m_Initialized;
      private int       m_MaxErrCount;
      private int       m_OldPage;
      private TOnUpdateWindow m_OnPaintCallback;
      private int       m_PageCount;
      private CPDF      m_PDF;
      private int       m_Resolution;
      private int       m_ScrollHMax;
      private int       m_ScrollVMax;
      private bool      m_ScrollWindow;
      private bool      m_ZoomMode;

      // Does this programming language really support Windows?

      [StructLayout(LayoutKind.Sequential, Pack=0)]
      public class TPoint
      {
         public int x;
         public int y;
      }

      [StructLayout(LayoutKind.Sequential, Pack=0)]
      public class TRect
      {
         public int Left;
         public int Top;
         public int Right;
         public int Bottom;
      }

      [StructLayout(LayoutKind.Sequential, Pack=0)]
      public class TSCROLLINFO
      {
          public int  cbSize;
          public uint fMask;
          public int  nMin;
          public int  nMax;
          public int  nPage;
          public int  nPos;
          public int  nTrackPos;
      }
      private const int ETO_OPAQUE       = 2;

      private const int FALSE            = 0;
      private const int TRUE             = 1;

      private const int SB_HORZ          = 0;
      private const int SB_VERT          = 1;
      private const int SB_BOTH          = 3;
      private const int SB_LINEUP        = 0;
      private const int SB_LINELEFT      = 0;
      private const int SB_LINEDOWN      = 1;
      private const int SB_LINERIGHT     = 1;
      private const int SB_PAGEUP        = 2;
      private const int SB_PAGEDOWN      = 3;
      private const int SB_THUMBTRACK    = 5;

      private const int ESB_ENABLE_BOTH  = 0;
      private const int ESB_DISABLE_BOTH = 3;

      private const int HORZRES    = 8;
      private const int LOGPIXELSX = 88;

      private const int SIF_RANGE           = 0x0001;
      private const int SIF_PAGE            = 0x0002;
      private const int SIF_POS             = 0x0004;
      private const int SIF_DISABLENOSCROLL = 0x0008;
      private const int SIF_TRACKPOS        = 0x0010;
      private const int SIF_ALL             = (SIF_RANGE | SIF_PAGE | SIF_POS | SIF_TRACKPOS);

      private const int WM_HSCROLL          = 0x0114;
      private const int WM_VSCROLL          = 0x0115;

      private const int VK_PRIOR            = 0x21;
      private const int VK_NEXT             = 0x22;
      private const int VK_END              = 0x23;
      private const int VK_HOME             = 0x24;
      private const int VK_LEFT             = 0x25;
      private const int VK_UP               = 0x26;
      private const int VK_RIGHT            = 0x27;
      private const int VK_DOWN             = 0x28;

      [DllImport("gdi32.dll", CharSet=CharSet.Auto)]private static extern uint   ExtTextOut(IntPtr DC, int x, int y, int Options, TRect Clip, String Value, uint Count, IntPtr Spacing);
      [DllImport("gdi32.dll", CharSet=CharSet.Ansi)]private static extern IntPtr GetCurrentPositionEx(IntPtr DC, TPoint p);
      [DllImport("gdi32.dll", CharSet=CharSet.Auto)]private static extern int    GetDeviceCaps(IntPtr DC, int Index);
      [DllImport("gdi32.dll", CharSet=CharSet.Ansi)]private static extern IntPtr LineTo(IntPtr DC, int x, int y);
      [DllImport("gdi32.dll", CharSet=CharSet.Ansi)]private static extern IntPtr MoveToEx(IntPtr DC, int x, int y, TPoint p);
      [DllImport("gdi32.dll", CharSet=CharSet.Ansi)]private static extern uint   SetBkColor(IntPtr DC, uint Color);

      [DllImport("user32.dll", CharSet=CharSet.Ansi)]private static extern int    EnableScrollBar(IntPtr hWnd, int Flags, int Arrows);
      [DllImport("user32.dll", CharSet=CharSet.Ansi)]private static extern IntPtr GetDC(IntPtr hWnd);
      [DllImport("user32.dll", CharSet=CharSet.Ansi)]private static extern int    GetScrollInfo(IntPtr hWnd, int nBar, TSCROLLINFO Info);
      [DllImport("user32.dll", CharSet=CharSet.Ansi)]private static extern IntPtr ReleaseDC(IntPtr hWnd, IntPtr DC);
      [DllImport("user32.dll", CharSet=CharSet.Auto)]private static extern int    SendMessage(IntPtr hWnd, int Msg, int wParam, int lParam);
      [DllImport("user32.dll", CharSet=CharSet.Ansi)]private static extern int    SetScrollInfo(IntPtr hWnd, int nBar, TSCROLLINFO Info, int Redraw);
      [DllImport("user32.dll", CharSet=CharSet.Ansi)]private static extern int    ShowScrollBar(IntPtr hWnd, int nBar, int Show);
   }
}
